const UNIT_FEAR: int = 3
const MAX_FEAR: int = 10 // So you don't die instantly when you reach 3 fear.

enum TileType
	BLANK
	FLOOR
	WALL
	WATER
	PLANT
	FOUNT
	ITEM
	ENEMY

struct TileData
	symbol: char
	name: string
	color: int

const terrain: array of TileData = {
	{' ', "void", TileType.BLANK},
	{'.', "a floor tile", TileType.FLOOR},
	{'+', "a wall", TileType.WALL},
	{'~', "a pool of water", TileType.WATER},
	{'=', "brambles", TileType.PLANT},
	{'*', "a fountain", TileType.FOUNT}
}

class Thing: Object
	level: weak Level
	x: int = -1
	y: int = -1
	
	symbol: char = '@'
	name: string = "thing"
	color: TileType = TileType.ITEM

	speed: int = 0
	light: int = 0
	glitter: int = 0

	next_thing: Thing?

	construct(l: Level)
		level = l
		if not (self in level)
			level.add(self)
	
	def move_to(x1: int, y1: int): bool
		if level.is_on_grid(x1, y1)
			x = x1
			y = y1
			return true
		else
			x = -1
			y = -1
			return false
	
	def a_name(): string
		vowels: array of char = {'a', 'e', 'i', 'o', 'u'}
		if name[0] in vowels
			return "an " + name
		else
			return "a " + name

	def to_string(): string
		return self.a_name()

enum Compass
	NORTH
	SOUTH
	EAST
	WEST
	CENTER

def advance(ref x: int, ref y: int, direction: Compass)
	case direction
		when Compass.NORTH do y -= 1
		when Compass.SOUTH do y += 1
		when Compass.EAST do x += 1
		when Compass.WEST do x -= 1

def inline abs(n: int): int
	if n < 0
		return -n
	else
		return n

def inline even(n: int): bool
	return n % 2 == 0

def inline odd(n: int): bool
	return n % 2 == 1

def roll_dice(count: int, sides: int): int
	var roll = 0
	for var i = 1 to count
		roll += Random.int_range(0, sides) + 1
	return roll

class Creature: Thing
	fear: int = 0
	
	base_speed: int = 0	
	action_points: int = 0
	last_heading: Compass = Compass.CENTER // For the AI system.

	event log(message: string)
	event progress()

	construct(l: Level)
		super(l)
	
	prop sight_radius: int
		get
			return light / 50 + 2

	def can_see(x1: int, y1: int): bool
		var can_see_x = abs(x - x1) <= sight_radius
		var can_see_y = abs(y - y1) <= sight_radius
		return can_see_x and can_see_y
		
	def can_enter(x: int, y: int): bool
		if level.is_on_grid(x, y)
			return level[x, y] == TileType.FLOOR
		else
			return false

	def is_way_clear(direction: Compass): bool
		var new_x = x
		var new_y = y
		
		advance(ref new_x, ref new_y, direction)

		if not level.is_on_grid(new_x, new_y)
			return false
		else if level.creature_at(new_x, new_y) != null
			return false
		else
			return can_enter(new_x, new_y)

	def walk(direction: Compass): bool
		var new_x = x
		var new_y = y
		
		advance(ref new_x, ref new_y, direction)

		if not level.is_on_grid(new_x, new_y)
			if direction == Compass.WEST
				log("There's no going back.")
				return false
			else if direction == Compass.EAST
				level.number += 1
				level.populate()
				if not (self in level)
					level.add(self)
				move_to(0, level.entry_line)
				fear -= UNIT_FEAR
				if fear < 0
					fear = 0
				progress()
				return true
			else
				log("You can't stray from the path.")
				return false
		else if level.creature_at(new_x, new_y) != null
			var creature = level.creature_at(new_x, new_y)
			var x1 = creature.x
			var y1 = creature.y
			creature.x = x
			creature.y = y
			x = x1
			y = y1
			action_points -= 5
			return true
		else if not can_enter(new_x, new_y)
			var t = level.terrain_at(new_x, new_y)
			log("The way is barred by "
				+ terrain[t].name + ".")
			return false
		else
			x = new_x
			y = new_y
			action_points -= 5
			return true
	
	def distance_to(thing: Thing): int
		return abs(x - thing.x) + abs(y - thing.y)
	
	def look(x: int, y: int)
		var msg = "You see: " + terrain[level[x, y]].name

		if not level.is_on_grid(x, y)
			return
		else if not can_see(x, y)
			log("You can't see that far right now.")
			return
		
		var i = level.content
		while i != null
			if i.x == x and i.y == y
				if i is self
					msg += ", yourself"
				else
					msg += ", " + i.name
			i = i.next_thing
		log(msg)
	
	def get_item_here()
		var item = level.thing_at(x, y)

		if item == null
			log("There's nothing portable here.")
			return
		
		light += item.light
		speed += item.speed
		glitter += item.glitter
		
		item.level.remove(item)
		log("You find " + item.name + ".")

class Level: Object
	const width: int = 80
	const height: int = 20
	level_gen: LevelGenerator

	number: int = 1

	tiles: array of TileType[,]
	entry_line: int = -1
	exit_line: int = 10

	content: Thing?

	init
		tiles = new array of TileType[height, width]
		level_gen = new LevelGenerator()
	
	def is_on_grid(x: int, y: int): bool
		return x >= 0 and y >= 0 and x < width and y < height
	
	def terrain_at(x: int, y: int): TileType
		if is_on_grid(x, y)
			return tiles[y, x]
		else
			return TileType.BLANK
	
	def thing_at(x: int, y: int): Thing?
		var i = content
		while i != null
			if not (i isa Creature) and i.x == x and i.y == y
				return i
			else
				i = i.next_thing
		return null
	
	def creature_at(x: int, y: int): Creature?
		var i = content
		while i != null
			if i isa Creature and i.x == x and i.y == y
				return (Creature?) i
			else
				i = i.next_thing
		return null

	def contains(thing: Thing): bool
		var i = content
		while i != null
			if i is thing
				return true
			else
				i = i.next_thing
		return false

	def add(thing: Thing)
		thing.next_thing = content
		content = thing
	
	def remove(thing: Thing)
		if content is thing
			content = thing.next_thing
		else
			var i = content.next_thing
			while i != null
				if i.next_thing is thing
					i.next_thing = thing.next_thing
					break
				else
					i = i.next_thing
	
	def new get(x: int, y: int): TileType
		return tiles[y, x]
	
	def new set(x: int, y: int, t: TileType)
		tiles[y, x] = t
	
	def populate()
		level_gen.populate(self)

class World: Object
	palace: Level
	player: Creature

	construct()
		palace = new Level()
		reset()

	def reset()
		palace.populate()
		player = new Creature(palace)
		player.base_speed = 100
		player.color = TileType.BLANK
		player.move_to(0, palace.entry_line)

	def update()
		var i = palace.content
		while i != null
			if i isa Creature
				var j = (Creature) i
				if j is player
					if j.light > 0
						j.light -= j.light / 150 + 1
					if j.speed > 0
						j.speed -= 1
				else if j.fear <= 0
					palace.remove(i)
					player.log(
						"The nightmare plays itself out.")
				else if j.distance_to(player) <= 1
					j.fear -= 1
					player.fear += 1
					player.log("Your fear grows.")

				if j.action_points <= 0
					var total = j.speed + j.base_speed
					j.action_points += total / 10
				else if not (j is player)
					run_ai(j)
					pass
			else if i.x == player.x and i.y == player.y
				player.get_item_here()

			i = i.next_thing

	def run_ai(mob: Creature)
		var x = mob.x
		var y = mob.y

		if mob.distance_to(player) == 1
			// Reset chase mode.
			mob.last_heading = Compass.CENTER
			return
		else if mob.x == player.x
			if mob.y < player.y
				mob.last_heading = Compass.SOUTH
			else if mob.y > player.y
				mob.last_heading = Compass.NORTH
		else if mob.y == player.y
			if mob.x < player.x
				mob.last_heading = Compass.EAST
			else if mob.x > player.x
				mob.last_heading = Compass.WEST
		else if mob.last_heading == Compass.CENTER
			// Pick any direction except for CENTER
			mob.last_heading = (Compass) Random.int_range(0, 4)

		advance(ref x, ref y, mob.last_heading)
		
		if mob.level.is_on_grid(x, y) and mob.can_enter(x, y)
			mob.walk(mob.last_heading)
		else
			mob.last_heading = Compass.CENTER

enum ItemType
	NOTHING
	LIGHT
	GLITTER
	SPEED
struct ItemData
	symbol: char
	name: string
	color: TileType

const items: array of ItemData = {
	{' ', "nothing", TileType.BLANK},
	{'!', "light", TileType.ITEM},
	{'$', "glitter", TileType.ITEM},
	{'?', "speed", TileType.ITEM}
}
const item_odds: array of ItemType = {
	ItemType.LIGHT, ItemType.LIGHT,
	ItemType.GLITTER, ItemType.GLITTER,
	ItemType.SPEED, ItemType.SPEED,
	ItemType.NOTHING, ItemType.NOTHING,
	ItemType.NOTHING, ItemType.NOTHING
}
const fear_odds: float = 0.6f;

def inline random_item(): ItemType
	return item_odds[Random.int_range(0, item_odds.length)]

def make_item(kind: ItemType, l: Level): Thing
	var item = new Thing(l)
	
	item.name = items[kind].name
	item.symbol = items[kind].symbol
	item.color = items[kind].color
	
	case kind
		when ItemType.LIGHT do item.light = 75
		when ItemType.SPEED do item.speed = 50
		when ItemType.GLITTER do item.glitter = roll_dice(5, 20)
	
	return item

def make_nightmare(l: Level): Creature
	var creature = new Creature(l)
	creature.name = "nightmare"
	creature.symbol = '&'
	creature.color = TileType.ENEMY
	creature.fear = UNIT_FEAR
	creature.base_speed = 150
	return creature

struct Bounds
	x1: int
	y1: int
	x2: int
	y2: int

class LevelGenerator: Object
	const cellw: int = 7
	const cellh: int = 5
	
	seed: uint32 = Random.next_int()
	
	def populate(level: Level)
		Random.set_seed(seed)
	
		for var x = 0 to (Level.width - 1)
			for var y = 0 to (Level.height - 1)
				level[x, y] = TileType.FLOOR
		for var x = 0 to (Level.width - 1)
			level.tiles[0, x] = TileType.WALL
			level.tiles[Level.height - 1, x] = TileType.WALL
		for var y = 0 to (Level.height - 1)
			level.tiles[y, 0] = TileType.WALL
			level.tiles[y, Level.width - 1] = TileType.WALL
		
		subdivide_wide(level, {1, 1, Level.width - 2, Level.height - 2})
		
		level.entry_line = level.exit_line
		level.tiles[level.entry_line, 0] = TileType.FLOOR
		level.tiles[level.entry_line, 1] = TileType.FLOOR
		level.exit_line = roll_dice(1, 18)
		level.tiles[level.exit_line, 79] = TileType.FLOOR
		level.tiles[level.exit_line, 78] = TileType.FLOOR

		level.content = null

		for var y = 1 to (Level.height - 2)
			for var x = 1 to (Level.width - 2)
				if level[x, y] != TileType.FLOOR
					continue
				else if Random.next_double() >= 0.03
					continue
				else
					var kind = random_item()
					if kind != ItemType.NOTHING
						var item = make_item(
							kind, level)
						item.move_to(x, y)
		
		for var y = 1 to (Level.height - 2)
			for var x = 1 to (Level.width - 2)
				if level[x, y] != TileType.FLOOR
					continue
				else if Random.next_double() >= 0.03
					continue
				else if Random.next_double() < fear_odds
					var c = make_nightmare(level)
					c.move_to(x, y)
		
		seed = Random.next_int()
	
	def subdivide_wide(level: Level, bounds: Bounds)
		var x = 0
		var w = bounds.x2 - bounds.x1 + 1
		var h = bounds.y2 - bounds.y1 + 1
		// You have to check both dimensions
		// or you'll get oddly skewed levels.
		if w < cellw or h < cellh or w == h
			furnish_room(level, bounds)
			return
		
		if w == 3
			x = bounds.x1 + 1
		else
			x = bounds.x1 + roll_dice(1, w - 2)
		for var y = bounds.y1 to bounds.y2
			level[x, y] = TileType.WALL

		subdivide_high(level, {bounds.x1, bounds.y1, x - 1, bounds.y2})
		subdivide_high(level, {x + 1, bounds.y1, bounds.x2, bounds.y2})
		
		if h >= 15
			var door_y = bounds.y1 + h / 3
			level.tiles[door_y, x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[door_y, x - 1] = TileType.FLOOR
			level.tiles[door_y, x + 1] = TileType.FLOOR
			door_y = bounds.y2 - h / 3
			level.tiles[door_y, x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[door_y, x - 1] = TileType.FLOOR
			level.tiles[door_y, x + 1] = TileType.FLOOR
		else
			var door_y = bounds.y1 + h / 2
			level.tiles[door_y, x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[door_y, x - 1] = TileType.FLOOR
			level.tiles[door_y, x + 1] = TileType.FLOOR

	def subdivide_high(level: Level, bounds: Bounds)
		var y = 0
		var w = bounds.x2 - bounds.x1 + 1
		var h = bounds.y2 - bounds.y1 + 1
		// You have to check both dimensions
		// or you'll get oddly skewed levels.
		if w < cellw or h < cellh or w == h
			furnish_room(level, bounds)
			return
		
		if h == 3
			y = bounds.y1 + 1
		else
			y = bounds.y1 + roll_dice(1, h - 2)
		for var x = bounds.x1 to bounds.x2
			level[x, y] = TileType.WALL

		subdivide_wide(level, {bounds.x1, bounds.y1, bounds.x2, y - 1})
		subdivide_wide(level, {bounds.x1, y + 1, bounds.x2, bounds.y2})

		if w >= 45
			var door_x = bounds.x1 + w / 4
			level.tiles[y, door_x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[y - 1, door_x] = TileType.FLOOR
			level.tiles[y + 1, door_x] = TileType.FLOOR
			door_x = bounds.x1 + w / 2
			level.tiles[y, door_x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[y - 1, door_x] = TileType.FLOOR
			level.tiles[y + 1, door_x] = TileType.FLOOR
			door_x = bounds.x2 - w / 4
		else if w >= 15
			var door_x = bounds.x1 + w / 3
			level.tiles[y, door_x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[y - 1, door_x] = TileType.FLOOR
			level.tiles[y + 1, door_x] = TileType.FLOOR
			door_x = bounds.x2 - w / 3
			level.tiles[y, door_x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[y - 1, door_x] = TileType.FLOOR
			level.tiles[y + 1, door_x] = TileType.FLOOR
		else
			var door_x = bounds.x1 + w / 2
			level.tiles[y, door_x] = TileType.FLOOR
			// Account for walls placed deeper into recursion.
			level.tiles[y - 1, door_x] = TileType.FLOOR
			level.tiles[y + 1, door_x] = TileType.FLOOR

	def furnish_room(level: Level, bounds: Bounds)
		var w = bounds.x2 - bounds.x1 + 1
		var h = bounds.y2 - bounds.y1 + 1

		if w == 3 and h == 3
			level[bounds.x1 + 1, bounds.y1 + 1] = TileType.FOUNT
		else if w == 3 and h > 2
			var y = 0
			for y = bounds.y1 to bounds.y2
				if even(y + bounds.x1 + 1)
					level[bounds.x1 + 1, y] = TileType.WALL
			if y == bounds.y2 and even(y + bounds.x1 + 1)
				level[bounds.x1 + 1, y] = TileType.FOUNT
		else if h == 3 and w > 2
			var x = 0
			for x = bounds.x1 to bounds.x2
				if even(bounds.y1 + 1 + x)
					level[x, bounds.y1 + 1] = TileType.WALL
			if x == bounds.x2 and even(bounds.y1 + 1 + x)
				level[x, bounds.y1 + 1] = TileType.FOUNT
		else if w == 4
			var symbol = TileType.BLANK
			if Random.next_double() < 0.5
				symbol = TileType.WATER
			else
				symbol = TileType.PLANT
			for var y = (bounds.y1 + 1) to (bounds.y2 - 1)
				level[bounds.x1 + 1, y] = symbol
				level[bounds.x2 - 1, y] = symbol
		else if h == 4
			var symbol = TileType.BLANK
			if Random.next_double() < 0.5
				symbol = TileType.WATER
			else
				symbol = TileType.PLANT
			for var x = (bounds.x1 + 1) to (bounds.x2 - 1)
				level[x, bounds.y1 + 1] = symbol
				level[x, bounds.y2 - 1] = symbol
		else if w == 5 and h == 5
			level[bounds.x1 + 1, bounds.y1 + 1] = TileType.FOUNT
			level[bounds.x1 + 3, bounds.y1 + 3] = TileType.FOUNT
			level[bounds.x1 + 3, bounds.y1 + 1] = TileType.FOUNT
			level[bounds.x1 + 1, bounds.y1 + 3] = TileType.FOUNT
		else if w == 5 and h > 2
			var y = 0
			for y = bounds.y1 to bounds.y2
				if even(y + bounds.x1 + 1)
					level[bounds.x1 + 1, y] = TileType.WALL
					level[bounds.x2 - 1, y] = TileType.WALL
			if y == bounds.y2 and even(y + bounds.x1 + 1)
				level[bounds.x1 + 1, y] = TileType.FOUNT
				level[bounds.x2 - 1, y] = TileType.FOUNT
		else if h == 5 and w > 2
			var x = 0
			for x = bounds.x1 to bounds.x2
				if even(bounds.y1 + 1 + x)
					level[x, bounds.y1 + 1] = TileType.WALL
					level[x, bounds.y2 - 1] = TileType.WALL
			if x == bounds.x2 and even(bounds.y1 + 1 + x)
				level[x, bounds.y1 + 1] = TileType.FOUNT
				level[x, bounds.y2 - 1] = TileType.FOUNT
		else if w == 7 and h == 7
			for var y = (bounds.y1 + 1) to (bounds.y1 + 2)
				for var x = (bounds.x1 + 1) to (bounds.x1 + 2)
					level[x, y] = TileType.PLANT
			for var y = (bounds.y1 + 1) to (bounds.y1 + 2)
				for var x = (bounds.x1 + 4) to (bounds.x1 + 5)
					level[x, y] = TileType.WATER
			for var y = (bounds.y1 + 4) to (bounds.y1 + 5)
				for var x = (bounds.x1 + 4) to (bounds.x1 + 5)
					level[x, y] = TileType.PLANT
			for var y = (bounds.y1 + 4) to (bounds.y1 + 5)
				for var x = (bounds.x1 + 1) to (bounds.x1 + 2)
					level[x, y] = TileType.WATER
		else if w > 5 and h > 5 and (odd(w) or odd(h))
			if odd(h)
				for var y = (bounds.y1 + 1) to (bounds.y2 - 1)
					if even((bounds.y1 + 1) - y)
						level[bounds.x1 + 1, y] = TileType.WALL
						level[bounds.x2 - 1, y] = TileType.WALL
			if odd(w)
				for var x = (bounds.x1 + 1) to (bounds.x2 - 1)
					if even((bounds.x1 + 1) - x)
						level[x, bounds.y1 + 1] = TileType.WALL
						level[x, bounds.y2 - 1] = TileType.WALL
			furnish_room(level, {
				bounds.x1 + 2,
				bounds.y1 + 2,
				bounds.x2 - 2,
				bounds.y2 - 2
			})
