uses
	Curses

init
	initscr()
	cbreak()
	nonl()
	noecho()
	stdscr.keypad(true)

	play_game()

	endwin()

def dir_keys(code: int): Compass
	t:Compass = Compass.CENTER
	case code
		when 'h' do t = Compass.WEST
		when 'j' do t = Compass.SOUTH
		when 'k' do t = Compass.NORTH
		when 'l' do t = Compass.EAST
		when Key.LEFT do t = Compass.WEST
		when Key.DOWN do t = Compass.SOUTH
		when Key.UP do t = Compass.NORTH
		when Key.RIGHT do t = Compass.EAST
	return t

const main_menu: array of string = {
	"[N]ew game",
	"[C]ontinue game",
	"[H]ow to play",
	"[B]est scores",
	"[Q]uit"
}

const title_banner: string = """
      __  _                                          _            _        
     /  || | o                  o                 \_|_)  o       | |       
     \__|| |   _|__|_  _   ,_       _  _    __,     |        __, | |   _|_ 
        ||/  |  |  |  |/  /  |  |  / |/ |  /  |    _|    |  /  | |/ \   |  
    (__/ |__/|_/|_/|_/|__/   |_/|_/  |  |_/\_/|/  (/\___/|_/\_/|/|   |_/|_/
                                             /|               /|           
                                             \|               \|           
"""

const end_banner: string = """
           _                                   _                     
          (_|   |                             | |                    
            |   |  __                    __,  | |   _             _  
            |   | /  \_|   |    |  |  |_/  |  |/_) |/    |   |  |/ \_
             \_/|/\__/  \_/|_/   \/ \/  \_/|_/| \_/|__/   \_/|_/|__/ 
               /|                                              /|    
               \|                                              \|    
                                            _                        
            o               |              | |                       
                _  _      __|   __,   ,_   | |   _  _    _   ,   ,   
            |  / |/ |    /  |  /  |  /  |  |/_) / |/ |  |/  / \_/ \_ 
            |_/  |  |_/  \_/|_/\_/|_/   |_/| \_/  |  |_/|__/ \/  \/ 
"""

const help_text: array of string = {
	"Get as far as you can, collecting glitter for score.",
	"Being right next to nightmares increases your fear.",
	"",
	"Arrow keys or h-j-k-l to move. '.' (period) to wait.",
	"'r' and a direction to run that way until stopped.",
	"",
	"? to show instructions during the game.",
	"",
	"Press 'x' and a digit to examine nearby items.",
	"You can also click the mouse for it, where available."
}

class HighScore
	name: string
	level: int
	glitter: int

	construct(n: string, l: int, g: int)
		name = n
		level = l
		glitter = g

	// Over long games, you pick up 2-3 glitter piles per level.
	prop score_value: int
		get
			return level * 125 + glitter

world: World
game_log: GameLog
highscores: GenericArray of HighScore

title_screen: Screen
game_screen: Screen
score_screen: Screen

active_screen: weak Screen

finished: bool = false

def play_game()
	mousemask(MouseMask.ALL_MOUSE_EVENTS, null)

	stdscr.leaveok(false)
	stdscr.idlok(true)
	stdscr.scrollok(true)
	stdscr.setscrreg(21, 23)

	if has_colors()
		start_color()
		init_pair(TileType.FLOOR, Color.MAGENTA, Color.BLACK)
		init_pair(TileType.WALL, Color.WHITE, Color.YELLOW)
		init_pair(TileType.WATER, Color.CYAN, Color.BLUE)
		init_pair(TileType.PLANT, Color.BLACK, Color.GREEN)
		init_pair(TileType.FOUNT, Color.CYAN, Color.BLACK)
		init_pair(TileType.ITEM, Color.YELLOW, Color.BLACK)
		init_pair(TileType.ENEMY, Color.RED, Color.BLACK)

	world = new World()
	game_log = new GameLog()
	highscores = new GenericArray of HighScore()

	title_screen = new TitleScreen()
	game_screen = new GameScreen()
	score_screen = new ScoreScreen()
	
	active_screen = title_screen
	
	while not finished
		var key = 0
		if (LINES < 24 or COLS < 80)
			erase()
			mvaddstr(0, 0, "80x24 or larger terminal required.")
			refresh()
		else
			stdscr.setscrreg(21, LINES - 1)
			active_screen.render()
		key = getch()
		if key == Key.RESIZE
			erase()
		else if key == 27
			finished = true
		else
			active_screen.handle_key(key)

def draw_dialog_bg(window: Window)
	window.bkgdset(' ' | Attribute.REVERSE)
	window.erase()
	window.box(0, 0)

def draw_help_dialog()
	var window = new Window(18, 66, 1, 7)
	draw_dialog_bg(window)
	window.mvaddstr(0, 1, "[ How to play ]")
	for var i = 0 to help_text.length
		window.mvaddstr(3 + i, 2, help_text[i])
	window.mvaddstr(17, 48, "[ Press any key ]")
	window.refresh()

def prompt(message: string): string
	var answer = ""
	var window = new Window(4, 36, 9, 22)
	draw_dialog_bg(window)
	window.mvaddstr(0, 1, "[ " + message/*[0:30]*/ + " ]")
	window.bkgdset(' ')
	window.mvaddstr(2, 3, "                              ")
	window.refresh()
	window.move(2, 3)
	echo()
	window.getnstr(answer, 30)
	noecho()
	return answer

def add_high_score(level: int, glitter: int)
	var name = prompt("High score! What's your name?")
	highscores.add(new HighScore(name, level, glitter))

// Sort higher scores before lower.
def compare_scores(a: HighScore, b: HighScore): int
	var score_a = a.score_value
	var score_b = b.score_value
	if score_a < score_b
		return 1
	else if score_a == score_b
		return 0
	else
		return -1

class GameLog: Object
	messages: GenericArray of string

	construct()
		messages = new GenericArray of string()
	
	def say(message: string)
		messages.add(message)
		while messages.length > 100
			messages.remove_index(0)

class abstract Screen: Object
	def render()
		render_viewport()
		
		standend()
		attron(Attribute.REVERSE)
		move(20, 0)
		for var i = 1 to 80
			addch(' ')
		mvaddstr(20, 0, status_text())
		attroff(Attribute.REVERSE)
			
		render_messages()

		refresh()

	def render_messages()
		var num_msg = LINES - 21
		move(21, 0)
		clrtobot()
		if game_log.messages.length >= num_msg
			var msg_from = game_log.messages.length - num_msg
			for var i = 0 to (num_msg - 1)
				mvaddstr(21 + i, 0,
					game_log.messages[msg_from + i])
		else
			for var i = 0 to (game_log.messages.length - 1)
				mvaddstr(21 + i, 0,
					game_log.messages[i])

	def abstract render_viewport()
	def abstract status_text(): string
	def abstract handle_key(key: int)

class TitleScreen: Screen
	def override render_viewport()
		for var i = 0 to 19
			move(i, 0)
			clrtoeol()
		
		attron(Attribute.BOLD)
		
		if has_colors()
			attron(COLOR_PAIR(TileType.FLOOR))
		mvaddstr(0, 0, title_banner)

		attroff(Attribute.BOLD)		
		
		if has_colors()
			standend()
			attron(COLOR_PAIR(TileType.BLANK))
		mvaddstr(9, 33, "A 2020 game by")
		if has_colors()
			attron(COLOR_PAIR(TileType.FOUNT))
		mvaddstr(11, 30, "www.notimetoplay.org")
		
		standend()
		
		for var i = 0 to (main_menu.length - 1)
			mvaddstr(i + 13, 33, main_menu[i])
	
	def override status_text(): string
		return ""

	def override handle_key(key: int)
		if key == 'q' or key == 27
			finished = true
		else if key == 'n'
			var gs = (GameScreen) game_screen
			world.reset()
			gs.clear_seen()
			world.player.log.connect(game_log.say)
			world.player.progress.connect(gs.clear_seen)
			active_screen = game_screen
		else if key == 'c'
			if world.player.fear >= MAX_FEAR
				var gs = (GameScreen) game_screen
				world.reset()
				gs.clear_seen()
				world.player.log.connect(game_log.say)
				world.player.progress.connect(gs.clear_seen)
			active_screen = game_screen
		else if key == 'h'
			draw_help_dialog()
			getch()
		else if key == 'b'
			active_screen = score_screen
		else
			flash()

class ScoreScreen: Screen
	def override render_viewport()
		for var i = 0 to 19
			move(i, 0)
			clrtoeol()

		mvaddstr(1, 2, "High Score Table")
		mvprintw(3, 2, "%-30s | %5s | %5s",
			"Player", "Level", "Score")
		mvhline(4, 2, Acs.HLINE, 46)
		for var i = 0 to (highscores.length - 1)
			mvprintw(5 + i, 2, "%-30s | %5d | %5d",
				highscores[i].name,
				highscores[i].level,
				highscores[i].glitter)
	
	def override status_text(): string
		return "[Esc] Menu"

	def override handle_key(key: int)
		if key == 'q' or key == 27
			active_screen = title_screen
		else
			flash()

/*struct Coords
	x: int
	y: int*/

def inline min(a: int, b: int): int
	if a < b
		return a
	else
		return b

def inline max(a: int, b: int): int
	if a > b
		return a
	else
		return b

delegate ThingFilter(t: Thing): bool

def inline filter_things(t: Thing): bool
	return not (t isa Creature)
def inline filter_creatures(t: Thing): bool
	return t isa Creature

class GameScreen: Screen
	seen: array of bool[,]
	look_mode: bool = false
	//target_squares: GenericArray of Coords?
	
	init
		seen = new array of bool[Level.height, Level.width]
	
	def override render_viewport()
		var level = world.palace
		var bounds = update_seen()
		
		attron(Attribute.DIM)
		for var y = 0 to (Level.height - 1)
			for var x = 0 to (Level.width - 1)
				if seen[y, x]
					render_tile(x, y, level[x, y])
				else
					mvaddch(y, x, ' ')
		attroff(Attribute.DIM)
		
		for var y = bounds.y1 to bounds.y2
			for var x = bounds.x1 to bounds.x2
				render_tile(x, y, level[x, y])
		
		//target_squares = new GenericArray of Coords?()
		render_things(level, filter_things)
		render_things(level, filter_creatures)
		
		if world.player.fear >= MAX_FEAR
			attron(Attribute.BOLD)
			if has_colors()
				attron(COLOR_PAIR(TileType.ENEMY))
			mvaddstr(2, 0, end_banner)

	def update_seen(): Bounds
		var sight_radius = world.player.sight_radius
		var x1 = max(0, world.player.x - sight_radius)
		var y1 = max(0, world.player.y - sight_radius)
		var x2 = min(Level.width - 1, world.player.x + sight_radius)
		var y2 = min(Level.height - 1, world.player.y + sight_radius)
		
		for var y = y1 to y2
			for var x = x1 to x2
				seen[y, x] = true
		
		return {x1, y1, x2, y2}
	
	def clear_seen()
		for var y = 0 to (Level.height - 1)
			for var x = 0 to (Level.width - 1)
				seen[y, x] = false		
	
	def render_tile(x: int, y: int, tile: TileType)
		if has_colors()
			attron(COLOR_PAIR(terrain[tile].color))
		mvaddch(y, x, terrain[tile].symbol)
		if has_colors()
			attroff(COLOR_PAIR(terrain[tile].color))
	
	def render_things(level: Level, to_show: ThingFilter)
		var i = level.content
		while not (i is null)
			if not to_show(i)
				i = i.next_thing
				continue
			else if not world.player.can_see(i.x, i.y)
				i = i.next_thing
				continue
			
			/*var symbol = i.char
			if i != world.player:
				square = (i.x, i.y)
				if square not in self.target_squares:
					self.target_squares.append(square)
				target_num = self.target_squares.index(square)
				if self.look_mode
					if target_num < 10
						symbol = str(target_num)*/
				
			attron(Attribute.BOLD)
			if has_colors()
				attron(COLOR_PAIR(i.color))
			mvaddch(i.y, i.x, i.symbol)
			if has_colors()
				attroff(COLOR_PAIR(i.color))
			
			i = i.next_thing

	def end_message(): string
		var msg = @"You made it to level $(world.palace.number),"
		msg += @" with $(world.player.glitter) glitter."
		return msg

	def override status_text(): string
		var msg = @"Level: $(world.palace.number)"
		msg += @" Glitter: $(world.player.glitter)"
		msg += @" Light: $(world.player.light)"
		msg += @" Speed: $(world.player.speed)"
		msg += " Fear: "
		for var i = 1 to (world.player.fear / UNIT_FEAR)
			msg += "&"
		return msg

	def override handle_key(key: int)
		if world.player.fear >= MAX_FEAR
			world.player.log(end_message())
			render_messages()
			add_high_score(\
				world.palace.number,\
				world.player.glitter)
			if highscores.length > 1
				highscores.sort_with_data(compare_scores)
				while highscores.length > 10
					highscores.remove_index(-1)
			look_mode = false
			active_screen = score_screen
			//world = null
			return
		else if key == 'q' or key == 27
			look_mode = false
			active_screen = title_screen
			return
		else if key == '?'
			look_mode = false
			draw_help_dialog()
			getch()
			return
		
		if world.player.action_points <= 0
			world.update()

		if dir_keys(key) != Compass.CENTER
			look_mode = false
			if world.player.walk(dir_keys(key))
				world.update()
		else if key == 'r'
			look_mode = false
			world.player.log("Run which way?")
			render_messages()
			var direction = getch()
			if dir_keys(direction) != Compass.CENTER
				var d = dir_keys(direction)
				timeout(500)
				while world.player.is_way_clear(d)
					world.player.walk(d)
					world.update()
					render()
					if getch() > -1
						break
				timeout(-1)
			else
				world.player.log("Canceled.")
		else if key == '.'
			look_mode = false
			world.player.log("You stand still.")
			world.update()
		//else if key == 'x'
		//	look_mode = not look_mode
		else if '0' <= key and key <= '9'
			/*var tnum = key - '0'
			if not self.look_mode:
				world.player.log("But you're not looking!")
			else if tnum >= target_squares.length
				world.player.log("Too few things around.")
			else if self.look_mode
				x, y = self.target_squares[tnum]
				world.player.look(x, y)
				self.look_mode = False*/
			pass
		else if key == Key.MOUSE
			me: MouseEvent
			getmouse(out me)
			if me.bstate == Button1.CLICKED
				world.player.look(me.x, me.y)
				look_mode = false
		else
			flash()
