<?php
// BrutalWiki: a wiki engine based on Brutalist principles
// 2021-07-23 Felix Pleşoianu <https://felix.plesoianu.ro/>
// Free and open source under the Artistic License 2.0

@include('wikiconf.php');

if (!defined('TIMEZONE')) define('TIMEZONE', 'UTC');
if (!defined('SITENAME')) define('SITENAME', 'BrutalWiki');
if (!defined('HOMEPAGE')) define('HOMEPAGE', 'Homepage');
if (!defined('TEMPLATE')) define('TEMPLATE', '');
if (!defined('DATETIME')) define('DATETIME', 'd F Y, H:i T');
if (!defined('DATABASE')) define('DATABASE', 'wiki.json');
if (!defined('PASSWORD')) define('PASSWORD', 'wikiwikiweb');

date_default_timezone_set(TIMEZONE);

$sample = <<<WIKI_SAMP
Hello, and welcome to BrutalWiki! This is the default homepage.

BrutalWiki is a wiki engine. You can add more content yourself.

=> https://en.wikipedia.org/wiki/Wiki_software What is a wiki engine?

You can add more pages, too, like this one:

-> Another page here

And of course as much text as you want. Have fun!
WIKI_SAMP;

$edit_form = <<<WIKI_EDIT
<form method="post">
 <input type="hidden" name="page_name" value="{{page_name}}">
 <textarea name="page_text" cols="64" rows="16">{{page_text}}</textarea>
 <br><label>Password <input type="password" name="secret"></label>
 <input type="submit" name="save" value="Save">
</form>
WIKI_EDIT;

function wiki_internal_link($text) {
	$link = urlencode(trim($text));
	$label = htmlspecialchars(trim($text));
	return "<a class='internal' href='?page=$link'>$label</a>";
}

function wiki_external_link($text) {
	$text = trim($text);
	$text = explode(" ", $text, 2);
	$link = $text[0];
	$label = htmlspecialchars(empty($text[1]) ? $text[0] : $text[1]);
	return "<a class='external' href='$link'>$label</a>";
}

function wiki_edit_link($text) {
	$link = urlencode(trim($text));
	$link = "?page=$link&amp;action=edit";
	return "<a href='$link' rel='nofollow'>Edit this page</a>";
}

function wiki_render($text) {
	$result = '';
	$text = preg_split("/\r\n|\r|\n/", $text);
	foreach ($text as $line) {
		if (empty($line)) {
			$result .= "<p>\n";
		} else if (substr($line, 0, 2) === '->') {
			$line = substr($line, 2);
			$result .= wiki_internal_link($line) . "<br>\n";
		} else if (substr($line, 0, 2) === '=>') {
			$line = substr($line, 2);
			$result .= wiki_external_link($line) . "<br>\n";
		} else {
			$result .= htmlspecialchars($line) . "\n";
		}
	}
	return $result;
}

function wiki_edit_box($pg, $text) {
	global $edit_form;
	return str_replace(
		array('{{page_name}}', '{{page_text}}'),
		array(htmlspecialchars($pg), htmlspecialchars($text)),
		$edit_form);
}

function check_password($form) {
	if (isset($form['secret']))
		return $form['secret'] === PASSWORD;
	else
		return false;
}

function wiki_save(&$db) {
	return @file_put_contents(DATABASE, json_encode($db));
}

function wiki_action_view($pg, &$db) {
	if (array_key_exists($pg, $db)) {
		return ['status' => 200,
			'title' => htmlspecialchars($pg),
			'body' => wiki_render($db[$pg]['content']),
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	} else {
		return ['status' => 404,
			'title' => htmlspecialchars($pg),
			'body' => 'Page not found: use edit link to create.',
			'modified' => date(DATETIME, time())];
	}
}

function wiki_action_edit($pg, &$db) {
	if (isset($_POST['save'])) {
		$pg = $_POST['page_name']; // Maybe do a sanity check here?
		if (!array_key_exists($pg, $db))
			$db[$pg] = ['created' => time(), 'modified' => time()];
		$db[$pg]['content'] = $_POST['page_text'];
		$last_edited = $db[$pg]['modified'];
		$db[$pg]['modified'] = time();
		if (!check_password($_POST)) {
			$body = '<p><output>Wrong password</output></p>'
				. wiki_edit_box($pg, $_POST['page_text']);
			$db[$pg]['modified'] = $last_edited;
		} else if (wiki_save($db)) {
			$body = "<p>Page saved: " . wiki_internal_link($pg);
		} else {
			$body = '<p><output>Save failed</output></p>'
				. wiki_edit_box($pg, $_POST['page_text']);
			$db[$pg]['modified'] = $last_edited;
		}
		return ['status' => 200,
			'title' => 'Editing ' . htmlspecialchars($pg),
			'body' => $body,
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	} else {
		if (!array_key_exists($pg, $db))
			$db[$pg] = ['content' => '', 'modified' => time()];
		return ['status' => 200,
			'title' => 'Editing ' . htmlspecialchars($pg),
			'body' => wiki_edit_box($pg, $db[$pg]['content']),
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	}
}

$wiki_actions = ['view' => 'wiki_action_view', 'edit' => 'wiki_action_edit'];

$database = array(HOMEPAGE => array(
	'content' => $sample, 'created' => time(), 'modified' => time()));

if (file_exists(DATABASE)) {
	$raw_data = @file_get_contents(DATABASE);
	if ($raw_data !== false) {
		$database = json_decode($raw_data, true);
		if ($database === null)
			exit("Error: database file is corrupt!");
	}
}

$page_name = empty($_GET['page']) ? HOMEPAGE : $_GET['page'];

if (empty($_GET['action']))
	$action = $wiki_actions['view'];
else if (array_key_exists($_GET['action'], $wiki_actions))
	$action = $wiki_actions[$_GET['action']];
else
	$action = $wiki_actions['view'];

$page_data = $action($page_name, $database);
$site_name = htmlspecialchars(SITENAME);

http_response_code($page_data['status']);
header('Content-type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<meta charset="utf-8">
<title><?php echo "$site_name: $page_data[title]"; ?></title>
<?php if (!empty(TEMPLATE)): ?>
 <link rel="stylesheet" href="<?php echo htmlspecialchars(TEMPLATE); ?>">
<?php endif; ?>

<header role="banner">
 <a href="?" class="logo"><?php echo $site_name; ?></a>
 <h1><?php echo $page_data['title']; ?></h1>
</header>

<hr>

<main role="main"><?php echo $page_data['body']; ?></main>

<hr>

<footer role="contentinfo">
 Last edited: <time><?php echo $page_data['modified']; ?></time>
 <?php if ($action === $wiki_actions['view']): ?>
  &bull; <?php echo wiki_edit_link($page_name); ?>
 <?php elseif ($action === $wiki_actions['edit']): ?>
  &bull; <?php echo wiki_internal_link($page_name); ?>
 <?php endif; ?>
</footer>
