<?php
// BrutalWiki: a wiki engine based on Brutalist principles
// 2021-07-23 Felix Pleşoianu <https://felix.plesoianu.ro/>
// Free and open source under the Artistic License 2.0

define('VERSION', '1.5.1');

$sample = <<<WIKI_SAMP
Hello, and welcome to BrutalWiki! This is the default homepage.

BrutalWiki is a wiki engine. You can add more content yourself.

=> https://en.wikipedia.org/wiki/Wiki_software What is a wiki engine?

You can add more pages, too, like this one:

-> Another page here

And of course as much text as you want. Have fun!
WIKI_SAMP;

$edit_form = <<<WIKI_EDIT
<form method="post">
 <input type="hidden" name="page_name" value="{{page_name}}">
 <textarea name="page_text" cols="64" rows="16">{{page_text}}</textarea>
 <br><label>Summary
  <input size="48" name="summary" value="{{summary}}"></label>
 <br><label>Password
  <input type="password" name="secret" value="{{last_pass}}"></label>
 <input type="submit" name="preview" value="Preview">
 <input type="submit" name="save" value="Save">
</form>
WIKI_EDIT;

$cleanup_form = <<<WIKI_CLEAN
<form method="post">
 <br><label>Password <input type="password" name="secret"></label>
 <input type="submit" name="clean" value="Run clean-up">
</form>
WIKI_CLEAN;

$search_form = <<<WIKI_SEARCH
<form method="get">
 <label>Search for
 <input type="search" name="target" value="{{text}}" required></label>
 <input type="hidden" name="action" value="search">
 <input type="submit" name="go" value="Go">
</form>
WIKI_SEARCH;

$new_page_form = <<<WIKI_NEW
<form method="get">
 <label>Page name <input name="page" required></label>
 <input type="hidden" name="action" value="edit">
 <input type="submit" name="create" value="Create">
</form>
WIKI_NEW;

$block_markup = array(
	'<<<' => '<blockquote>', '>>>' => '</blockquote>',
	'[quote]' => '<blockquote>', '[/quote]' => '</blockquote>',
	'(((' => '<ul>', ')))' => '</ul>',
	'[list]' => '<ul>', '[/list]' => '</ul>',
	'[[[' => '<ol>', ']]]' => '</ol>',
	'---' => '<hr>', '----' => '<hr>', '-----' => '<hr>',
	'{|' => '<table>', ' |-' => '<tr>', ' |}' => '</table>',
	'[left]' => '<div class="left" style="Float:left;Clear:left;">',
	'[right]' => '<div class="right" style="Float:right;Clear:right;">',
	'[center]' => '<div class="center" style="Text-align:center;">',
	'[back]' => '</div>');

$prefix_markup = array(
	'-' => 'li', '*' => 'li', '+' => 'li',
	'####' => 'h4', '###' => 'h3', '##' => 'h2',
	' |+' => 'caption', ' !' => 'th', ' |' => 'td');

$inline_markup = array(
	'[i]' => '<i>', '[/i]' => '</i>',
	'[b]' => '<b>', '[/b]' => '</b>',
	'[s]' => '<s>', '[/s]' => '</s>',
	'&lt;&lt;' => '<q>', '&gt;&gt;' => '</q>',
	'[q]' => '<q>', '[/q]' => '</q>',
	'{{' => '<code>', '}}' => '</code>',
	'[c]' => '<code>', '[/c]' => '</code>',
	'((' => '<small>', '))' => '</small>',
	'\\\\' => '<br>', '-- ' => '&mdash; ',
	// Just brute-force character entities.
	'&amp;mdash;' => '&mdash;', '&amp;bull;' => '&bull;');

$wiki_actions = array(
	'view' => 'view_action',
	'edit' => 'edit_action',
	'rc' => 'rc_action',
	'search' => 'search_action',
	'index' => 'index_action',
	'stats' => 'stats_action',
	'clean' => 'clean_action');

@include_once('wikiconf.php');

if (!defined('TIMEZONE')) define('TIMEZONE', 'UTC');
if (!defined('SITENAME')) define('SITENAME', 'BrutalWiki');
if (!defined('HOMEPAGE')) define('HOMEPAGE', 'Homepage');
if (!defined('MAINMENU')) define('MAINMENU', 'Menubar');
if (!defined('PAGEFOOT')) define('PAGEFOOT', 'Footer');
if (!defined('TEMPLATE')) define('TEMPLATE', '');
if (!defined('VIEWPORT')) define('VIEWPORT', '');
if (!defined('LOGO_URL')) define('LOGO_URL', '');
if (!defined('ICON_URL')) define('ICON_URL', '');
if (!defined('DATETIME')) define('DATETIME', 'd M Y, H:i T');
if (!defined('DATEONLY')) define('DATEONLY', 'd F Y');
if (!defined('RC_COUNT')) define('RC_COUNT', 50);
if (!defined('DATABASE')) define('DATABASE', 'wiki.json');
if (!defined('PASSWORD')) define('PASSWORD', 'wikiwikiweb');
if (!defined('READ_ONLY')) define('READ_ONLY', false);

date_default_timezone_set(TIMEZONE);

function render_prefix($text) {
	global $prefix_markup;
	foreach ($prefix_markup as $mark => $tag) {
		if (strpos($text, $mark) === 0) {
			$offset = strlen($mark);
			$text = substr($text, $offset);
			$text = render_inline(trim($text));
			return "<$tag>$text</$tag>";
		}
	}
	return render_inline($text);
}

function render_inline($text) {
	global $inline_markup;
	return str_ireplace(
		array_keys($inline_markup),
		array_values($inline_markup),
		htmlspecialchars($text));
}

function internal_link($text, &$db = null) {
	$text = trim($text);
	$link = urlencode($text);
	$label = render_inline($text);
	if ($db === null || array_key_exists($text, $db))
		return "<a class='internal' href='?page=$link'>$label</a>";
	else
		return "<a class='missing' href='?page=$link'>$label?</a>";
}

function external_link($text) {
	$text = trim($text);
	$text = explode(" ", $text, 2);
	$link = htmlspecialchars($text[0]);
	$label = render_inline(empty($text[1]) ? $text[0] : $text[1]);
	return "<a class='external' href='$link'>$label</a>";
}

function image_link($text) {
	$text = trim($text);
	$text = explode(" ", $text, 2);
	$link = htmlspecialchars($text[0]);
	$alt = empty($text[1]) ? "" : htmlspecialchars($text[1]);
	return "<img src='$link' alt='$alt'>";
}

function edit_link($text) {
	$link = urlencode(trim($text));
	$link = "?page=$link&amp;action=edit";
	return "<a href='$link' rel='nofollow'>Edit this page</a>";
}

function render_markup($text, &$db = null) {
	global $block_markup;
	$result = '';
	$preformatted = false;
	$code_block = false;
	$text = preg_split("/\r\n|\r|\n/", $text);
	foreach ($text as $line) {
		if ($preformatted) {
			if ($line === '}}}') {
				$result .= "</pre>\n";
				$preformatted = false;
			} else {
				$result .= htmlspecialchars($line) . "\n";
			}
		} else if ($code_block) {
			if ($line === '[/code]') {
				$result .= "</code></pre>\n";
				$code_block = false;
			} else {
				$result .= htmlspecialchars($line) . "\n";
			}
		} else if (empty($line)) {
			$result .= "<p>\n";
		} else if (substr($line, 0, 2) === '->') {
			$line = substr($line, 2);
			$result .= internal_link($line, $db) . "<br>\n";
		} else if (substr($line, 0, 2) === '=>') {
			$line = substr($line, 2);
			$result .= external_link($line) . "<br>\n";
		} else if (substr($line, 0, 2) === '~>') {
			$line = substr($line, 2);
			$result .= image_link($line) . "\n";
		} else if (substr($line, 0, 3) === '{{{') {
			$result .= "<pre>"; // No newline here.
			$preformatted = true;
		} else if (substr($line, 0, 6) === '[code]') {
			$result .= "<pre><code>"; // No newline here.
			$code_block = true;
		} else if (substr($line, 0, 2) === '%%') {
			$line = htmlspecialchars(substr($line, 2));
			$result .= "<!--$line-->\n";
		} else if (array_key_exists($line, $block_markup)) {
			$result .= $block_markup[$line] . "\n";
		} else {
			if (empty($result)) $result .= "<p>\n";
			$result .= render_prefix($line);
		}
	}
	return $result;
}

function parse_links($text) {
	$link_list = array();
	$text = preg_split("/\r\n|\r|\n/", $text);
	foreach ($text as $line) {
		if (substr($line, 0, 2) === '->') {
			$line = substr($line, 2);
			$link_list[] = internal_link($line);
		} else if (substr($line, 0, 2) === '=>') {
			$line = substr($line, 2);
			$link_list[] = external_link($line);
		}
	}
	return $link_list;
}

function empty_pages(&$db) {
	$emptied = array();
	foreach ($db as $title => $page)
		if (empty($page['content']))
			$emptied[] = $title;
	return $emptied;
}

function page_sizes(&$db) {
	$size_list = array();
	foreach ($db as $title => $page)
		$size_list[$title] = strlen($page['content']);
	return $size_list;
}

function make_list($items) {
	if (count($items) > 0) {
		$result = "<ul>\n";
		foreach ($items as $i)
			$result .= " <li>$i</li>\n";
		$result .= "</ul>\n";
		return $result;
	} else {
		return "";
	}
}

function edit_box($pg, $text) {
	global $edit_form;
	$summary = isset($_POST['summary']) ? $_POST['summary'] : '';
	$pass = isset($_POST['secret']) ? $_POST['secret'] : '';
	$vars = array(
		'{{page_name}}' => htmlspecialchars($pg),
		'{{page_text}}' => htmlspecialchars($text),
		'{{summary}}' => htmlspecialchars($summary),
		'{{last_pass}}' => htmlspecialchars($pass));
	return str_replace(array_keys($vars), array_values($vars), $edit_form);
}

function search_box($text) {
	global $search_form;
	return str_replace(
		['{{text}}'],
		[htmlspecialchars($text)],
		$search_form);
}

function run_search($text, &$db) {
	$titles = array(); $others = array();
	foreach ($db as $title => $page) {
		if (stripos($title, $text) !== false)
			$titles[] = $title;
		if (stripos($page['content'], $text) !== false)
			$others[] = $title;
	}
	return array($titles, $others);
}

function check_password($form) {
	if (isset($form['secret']))
		return $form['secret'] === PASSWORD;
	else
		return false;
}

function save_data(&$db) {
	return @file_put_contents(DATABASE, json_encode($db));
}

function view_action($pg, &$db) {
	if (array_key_exists($pg, $db)) {
		return ['status' => 200,
			'title' => htmlspecialchars($pg),
			'body' => render_markup($db[$pg]['content'], $db),
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	} else {
		return ['status' => 404,
			'title' => htmlspecialchars($pg),
			'body' => 'Page not found: use edit link to create.',
			'modified' => date(DATETIME, time())];
	}
}

function edit_action($pg, &$db) {
	if (isset($_POST['save'])) {
		$pg = $_POST['page_name']; // Maybe do a sanity check here?
		if (!array_key_exists($pg, $db))
			$db[$pg] = ['created' => time(), 'modified' => time()];
		$db[$pg]['content'] = $_POST['page_text'];
		$db[$pg]['summary'] = $_POST['summary'];
		$last_edited = $db[$pg]['modified'];
		$db[$pg]['modified'] = time();
		if (!check_password($_POST)) {
			$body = '<p><b>Wrong password</b></p>'
				. edit_box($pg, $_POST['page_text']);
			$db[$pg]['modified'] = $last_edited;
			sleep(5); // Throttle response to slow down bots.
		} else if (save_data($db)) {
			$body = "<p>Page saved: " . internal_link($pg);
		} else {
			$body = '<p><b>Save failed</b></p>'
				. edit_box($pg, $_POST['page_text']);
			$db[$pg]['modified'] = $last_edited;
		}
		return ['status' => 200,
			'title' => 'Editing ' . htmlspecialchars($pg),
			'body' => $body,
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	} else if (isset($_POST['preview'])) {
		return ['status' => 200,
			'title' => 'Editing ' . htmlspecialchars($pg),
			'body' => '<p><b>Preview:</b></p>'
				. render_markup($_POST['page_text'], $db)
				. edit_box($pg, $_POST['page_text']),
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	} else {
		if (!array_key_exists($pg, $db))
			$db[$pg] = ['content' => '', 'modified' => time()];
		return ['status' => 200,
			'title' => 'Editing ' . htmlspecialchars($pg),
			'body' => edit_box($pg, $db[$pg]['content']),
			'modified' => date(DATETIME, $db[$pg]['modified'])];
	}
}

function rc_action($pg, &$db) {
	$page_times = array();
	foreach ($db as $page_name => $data)
		$page_times[$page_name] = $data['modified'];
	arsort($page_times, SORT_NUMERIC);
	$page_times = array_slice($page_times, 0, RC_COUNT);
	$last_date = '';
	$body = "<dl>\n";
	foreach ($page_times as $page_name => $modified) {
		$page_date = htmlspecialchars(date(DATEONLY, $modified));
		if ($page_date !== $last_date) {
			$last_date = $page_date;
			$body .= " <dt><time>$last_date</time></dt>\n";
		}
		$page_link = internal_link($page_name);
		if (empty($db[$page_name]['summary'])) {
			$body .= " <dd>$page_link</dd>\n";
		} else {
			$summary = htmlspecialchars(
				$db[$page_name]['summary']);
			$body .= " <dd>$page_link: $summary</dd>\n";
		}
	}
	$body .= "</dl>\n";
	return ['status' => 200,
		'title' => 'Recent changes',
		'body' => $body,
		'modified' => date(DATETIME)];
}

function search_action($pg, &$db) {
	$body = '';
	if (isset($_GET['target'])) {
		$body .= search_box($_GET['target']) . "\n\n";
		list($titles, $others) = run_search($_GET['target'], $db);
		$body .= "<p>Title matches:</p>\n"
			. make_list(array_map('internal_link', $titles))
			. "<p>Other matches:</p>\n"
			. make_list(array_map('internal_link', $others));
	} else {
		global $new_page_form;
		$body .= search_box('') . "\n\n"
			. make_list([
				"<a href='?action=index'>All pages</a>",
				"<a href='?action=stats'>Statistics</a>"])
			. "\n<p>Special pages:</p>\n\n"
			. make_list([
				'Homepage: ' . internal_link(HOMEPAGE),
				'Main menu: ' . internal_link(MAINMENU),
				'Page footer: ' . internal_link(PAGEFOOT)]);
		if (!READ_ONLY)
			$body .= "\n<p>Or make a new page:</p>$new_page_form";
	}
	return ['status' => 200,
		'title' => 'Search ' . htmlspecialchars(SITENAME),
		'body' => $body,
		'modified' => date(DATETIME)];
}

function index_action($pg, &$db) {
	$size_list = page_sizes($db);
	ksort($size_list, SORT_STRING);
	$body = "<ol>\n";
	foreach ($size_list as $title => $size) {
		$ln = internal_link($title);
		$body .= " <li>$ln ($size bytes)</li>\n";
	}
	$body .= "</ol>\n\n";
	$body .= sprintf("<p>%d pages, %d bytes total</p>\n",
		count($size_list), array_sum($size_list));
	return ['status' => 200,
		'title' => 'All pages',
		'body' => $body,
		'modified' => date(DATETIME)];
}

function stats_action($pg, &$db) {
	$size_list = page_sizes($db);
	$num = count($size_list);
	$total = array_sum($size_list);
	$avg = round($total / $num);
	$emptied = count(empty_pages($db));
	$body = '<p>BrutalWiki version: ' . VERSION . "</p>\n\n";
	$body .= "<p>$num pages, $total bytes total, $avg average</p>\n\n";
	if ($emptied === 1)
		$body .= "<p>$emptied empty page";
	else
		$body .= "<p>$emptied empty pages";
	if ($emptied > 0 && !READ_ONLY)
		$body .= " <a href='?action=clean' rel='nofollow'>Clean up</a>";
	return ['status' => 200,
		'title' => 'Statistics',
		'body' => $body,
		'modified' => date(DATETIME)];
}

function clean_action($pg, &$db) {
	global $cleanup_form;
	if (isset($_POST['clean'])) {
		$emptied = empty_pages($db); 
		foreach ($emptied as $title)
			unset($db[$title]);
		if (!check_password($_POST)) {
			$body = "<p><b>Wrong password</b></p>\n$cleanup_form";
		} else if (save_data($db)) {
			$body = '<p>Pages cleaned up: ' . count($emptied);
		} else {
			$body = "<p><b>Save failed</b></p>\n$cleanup_form";
		}
		return ['status' => 200,
			'title' => 'Page cleanup',
			'body' => $body,
			'modified' => date(DATETIME)];
	} else {
		$emptied = count(empty_pages($db));
		if ($emptied === 1)
			$body = "<p>$emptied empty page</p>\n\n";
		else
			$body = "<p>$emptied empty pages</p>\n\n";
		return ['status' => 200,
			'title' => 'Page cleanup',
			'body' => $body . $cleanup_form,
			'modified' => date(DATETIME)];
	}
}

if (READ_ONLY) {
	unset($wiki_actions['edit']);
	unset($wiki_actions['clean']);
}

$database = array(HOMEPAGE => array(
	'content' => $sample, 'created' => time(), 'modified' => time()));

if (file_exists(DATABASE)) {
	$raw_data = @file_get_contents(DATABASE);
	if ($raw_data !== false) {
		$database = json_decode($raw_data, true);
		if ($database === null)
			exit("Error: database file is corrupt!");
	}
}

if (!empty(MAINMENU) && array_key_exists(MAINMENU, $database)) {
	$main_menu = $database[MAINMENU]['content'];
	$main_menu = implode(' &bull; ', parse_links($main_menu));
} else {
	$main_menu = '';
}

if (!empty(PAGEFOOT) && array_key_exists(PAGEFOOT, $database)) {
	$page_foot = $database[PAGEFOOT]['content'];
	$page_foot = render_markup($page_foot);
} else {
	$page_foot = '';
}

if (empty($_GET['action']))
	$action = $wiki_actions['view'];
else if (array_key_exists($_GET['action'], $wiki_actions))
	$action = $wiki_actions[$_GET['action']];
else
	$action = $wiki_actions['view'];

$page_name = empty($_GET['page']) ? HOMEPAGE : $_GET['page'];
$page_data = $action($page_name, $database);
$site_name = htmlspecialchars(SITENAME);

http_response_code($page_data['status']);
header('Content-type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<meta charset="utf-8">
<title><?php echo "$site_name: $page_data[title]"; ?></title>
<?php if (!empty(TEMPLATE)): ?>
 <link rel="stylesheet" href="<?php echo htmlspecialchars(TEMPLATE); ?>">
<?php endif; ?>
<?php if (!empty(ICON_URL)): ?>
 <link rel="icon" href="<?php echo htmlspecialchars(ICON_URL); ?>">
<?php endif; ?>
<?php if (!empty(VIEWPORT)): ?>
 <meta name="viewport" content="<?php echo htmlspecialchars(VIEWPORT); ?>">
<?php endif; ?>
<meta property="og:type" content="website">
<meta property="og:site_name" content="<?php echo $site_name; ?>">
<meta property="og:title" content="<?php echo $page_data['title']; ?>">

<header role="banner">
 <?php if (!empty(LOGO_URL)): ?>
  <a href="?" class="logo"><img
     src="<?php echo htmlspecialchars(LOGO_URL); ?>"
     alt="<?php echo htmlspecialchars(SITENAME); ?>"></a>
 <?php else: ?>
  <a href="?" class="logo"><?php echo $site_name; ?></a>
 <?php endif; ?>
 <h1><?php echo $page_data['title']; ?></h1>
</header>

<?php if (!empty($main_menu)) echo "<nav>$main_menu</nav>\n"; ?>

<hr>

<main role="main"><?php echo $page_data['body']; ?></main>

<hr>

<footer role="contentinfo">
 Last edited: <time><?php echo $page_data['modified']; ?></time>
 <?php if (READ_ONLY): ?>
  &bull; read-only mode
 <?php elseif ($action === $wiki_actions['view']): ?>
  &bull; <?php echo edit_link($page_name); ?>
 <?php elseif ($action === $wiki_actions['edit']): ?>
  &bull; <?php echo internal_link($page_name); ?>
 <?php endif; ?>
 &bull; <a href="?action=rc">Recent changes</a>
 &bull; <a href="?action=search">Search</a>
 
 <?php if (!empty($page_foot)) echo "<div class='custom'>$page_foot</div>\n"; ?>
</footer>
