#!/usr/bin/env python3
# 
# JourNote: Proof of concept journal app with a text user interface.
# No Time To Play, 2026
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

"proof of concept journal app with a text user interface"

import subprocess
import shutil
import time
import sys
import os
import os.path
import glob

import argparse
import configparser

version_string = "JourNote 2.1"

about_text = """
{version_string}
by No Time To Play
MIT License
"""

DLG_OK = 0
DLG_CANCEL = 1
DLG_YES = 0
DLG_NO = 1
DLG_HELP = 2
DLG_CLOSED = 255

file_ext = ".txt"
file_limit = 30
date_format = "%Y-%m-%d" # Will become part of filenames.
week_start : int|str = "Mo"

notes_file = "notes" # Sans extension
config_file = "journote.ini"
archive_dir = "archive"

config = configparser.ConfigParser(interpolation=None)

TITLE = "JourNote"
BACKTITLE = "JourNote: Little TUI journaling tool"

def config_home() -> str:
	"""Get storage dir for user config files as per the XDG spec."""
	if os.getenv("XDG_CONFIG_HOME"):
		return str(os.getenv("XDG_CONFIG_HOME"))
	elif sys.platform.startswith("haiku"):
		result = subprocess.run(
			["finddir", "B_USER_SETTINGS_DIRECTORY"],
			capture_output=True)
		return result.stdout.decode()
	else:
		return os.path.join(str(os.getenv("HOME")), ".config")

def dialog(control, **options):
	if "title" not in options:
		options["title"] = TITLE
	if "backtitle" not in options:
		options["backtitle"] = BACKTITLE
	return raw_dialog(["dialog"], control, options)

def raw_dialog(cmdline, control, options):
	for i in options:
		cmdline.append("--" + str(i).replace("_", "-"))
		if options[i] != None:
			cmdline.append(str(options[i]))
	if control[0] == "--checklist":
		cmdline.append("--separate-output")

	proc = subprocess.Popen(cmdline + control, stderr = subprocess.PIPE)
	_, output = proc.communicate()
	output = output.decode()

	if control[0] == "--editbox":
		pass # Return text file contents as is.
	elif control[0] == "--checklist":
		# Return output already parsed.
		output = output.rstrip().splitlines()
	else:
		output = output.rstrip() # Trim the newline after the tag.
	
	return proc.returncode, output

def editbox(path):
	return ["--editbox", str(path), "0", "0"]

def inputbox(text, init = ""):
	return ["--inputbox", str(text), "0", "0", str(init)]

def yesno(text):
	return ["--yesno", str(text), "0", "0"]

def msgbox(text):
	return ["--msgbox", str(text), "0", "0"]

def menu(text, items):
	args = ["--menu", str(text), "0", "0", "0"]
	for tag, label in items:
		args.append(str(tag))
		args.append(str(label))
	return args

def calendar(text):
	return ["--calendar", str(text), "0", "0"]

def checklist(text, items):
	args = ["--checklist", str(text), "0", "0", "0"]
	for tag, label, state in items:
		args.append(str(tag))
		args.append(str(label))
		if state:
			args.append("on")
		else:
			args.append("off")
	return args

def radiolist(text, entries):
	args = ["--radiolist", str(text), "0", "0", "0"]
	for tag, label, state in entries:
		args.append(str(tag))
		args.append(str(label))
		if state:
			args.append("on")
		else:
			args.append("off")
	return args

def rangebox(text, min_val, max_val, default):
	return ["--rangebox", str(text), "0", "0",
		str(int(min_val)), str(int(max_val)), str(int(default))]

main_menu = [("today", "Today's entry"), ("cal", "Calendar"),
	("notes", "Note file"), ("recent", "Recent entries"),
	("search", "Search entries"), ("archive", "Archive entries"),
	("opt", "Option menu"),	("version", "Version info")]

option_menu = [("ext", "File extension"), ("limit", "Limit entries"),
	("date", "Date format"), ("week", "Week start"),
	("pwd", "Print work dir"), ("save", "Save options")]

ext_menu = [(".txt", "Plain text"), (".md", "Markdown"), (".org", "Org Mode")]

date_menu = [("%Y-%m-%d", "Daily entries"), ("%Y-W%V", "Weekly entries"),
	("%Y-%m", "Monthly entries")]

week_menu = [(0, "Sunday"), (1, "Monday"), (2, "Tuesday"),
	(3, "Wednesday"), (4, "Thursday"), (5, "Friday"), (6, "Saturday")]

def edit_file(fn: str) -> None:
	if os.getenv("EDITOR"):
		subprocess.run([str(os.getenv("EDITOR")), fn])
	elif os.getenv("VISUAL"):
		subprocess.run([str(os.getenv("VISUAL")), fn])
	else:
		subprocess.run(["touch", fn])
		code, output = dialog(editbox(fn), title=fn)
		if code == 0:
			with open(fn, "w") as f:
				f.write(output)

def find_in_files(text):
	files = glob.glob("*" + file_ext)
	if len(files) == 0:
		return []
	cmd = ["grep", "-i", "-l", text] + files
	proc = subprocess.run(cmd, capture_output=True)
	return proc.stdout.decode().split()

def file_menu(files):
	items = []
	
	for i in files:
		fn, _ = os.path.splitext(i)
		items.append((i, fn))
	
	return items

def file_checklist(files):
	items = []
	
	for i in files:
		fn, _ = os.path.splitext(i)
		items.append((i, fn, False))
	
	return items

def radio_menu(entries, default):
	items = []
	
	for i, j in entries:
		if i == default:
			items.append((i, j, True))
		else:
			items.append((i, j, False))
	
	return items

def handle_recent(entries):
	while True:
		code, output = dialog(
			menu("Pick an entry", entries),
			title="Recent entries", no_tags=None,
			hline="Showing newest " + str(file_limit),
			ok_label="Edit", cancel_label="Back")
		if code == DLG_OK:
			edit_file(output)
		else:
			break

def handle_search(text, entries):
	while True:
		code, output = dialog(
			menu(text + " found in:", entries),
			title="Search results", no_tags=None,
			hline="Showing at most " + str(file_limit),
			ok_label="Edit", cancel_label="Back")
		if code == DLG_OK:
			edit_file(output)
		else:
			break

def handle_options():
	global file_ext, file_limit, date_format, week_start

	while True:
		code, output = dialog(
			menu("Pick an operation", option_menu),
			title="Option menu", no_tags=None,
			ok_label="Edit", cancel_label="Back")
		if code == DLG_CANCEL:
			break
		elif output == "ext":
			items = radio_menu(ext_menu, file_ext)
			code, output = dialog(
				radiolist("File type for journals", items),
				ok_label="Set", no_tags=None,
				title="File extension")
			if code == DLG_OK and output != "":
				file_ext = output
		elif output == "limit":
			code, output = dialog(
				rangebox("Max entries to show",
					10, 100, file_limit),
				title="Limit entries")
			if code == DLG_OK:
				file_limit = int(output)
		elif output == "date":
			items = radio_menu(date_menu, date_format)
			code, output = dialog(
				radiolist("Format file names as", items),
				ok_label="Set", no_tags=None,
				title="Date format")
			if code == DLG_OK and output != "":
				date_format = output
		elif output == "week":
			items = radio_menu(week_menu, week_start)
			code, output = dialog(
				radiolist("First weekday in calendar", items),
				ok_label="Set", title="Week start")
			if code == DLG_OK and output != "":
				week_start = int(output)
		elif output == "pwd":
			dialog(msgbox(os.getcwd()),
				title="Current working directory")
		elif output == "save":
			if not config.has_section("JourNote"):
				config.add_section("JourNote")
			config["JourNote"]["file_ext"] = file_ext
			config["JourNote"]["file_limit"] = str(file_limit)
			config["JourNote"]["date_format"] = date_format
			config["JourNote"]["week_start"] = str(week_start)
			try:
				with open(config_file, "w") as f:
					config.write(f)
				dialog(msgbox("Options saved"))
			except configparser.Error as e:
				dialog(msgbox(e), title="Config error")

work_dir = os.path.join(config_home(), "journote")

cmdline = argparse.ArgumentParser(
	description="Proof of concept journal app with a text user interface.")
cmdline.add_argument("-v", "--version", action="version",
	version=version_string)
cmdline.add_argument("-t", "--today", action='store_true',
	help="open today's entry then quit")
cmdline.add_argument("-n", "--notes", action='store_true',
	help="open notes file then quit")
cmdline.add_argument("workdir", nargs="?",
	help="work directory (default:" + work_dir + ")")

args = cmdline.parse_args()

if args.workdir != None:
	try:
		os.chdir(args.workdir)
	except OSError as e:
		print("Can't change directory: " + str(e), file=sys.stderr)
		sys.exit(2)
elif os.getenv("JNDIR"):
	try:
		os.chdir(str(os.getenv("JNDIR")))
	except OSError as e:
		print("Can't change directory: " + str(e), file=sys.stderr)
		sys.exit(2)
else:
	try:
		os.makedirs(work_dir, exist_ok = True)
		os.chdir(work_dir)
	except OSError as e:
		print("Can't change directory: " + str(e), file=sys.stderr)
		sys.exit(2)

if shutil.which("dialog") == None:
	print("Please install the dialog utility.", file=sys.stderr)
	sys.exit(1)

try:
	config.read(config_file)
	file_ext = config.get("JourNote", "file_ext", fallback=".txt")
	file_limit = config.getint("JourNote", "file_limit", fallback=30)
	date_format = config.get("JourNote", "date_format", fallback="%Y-%m-%d")
	week_start = config.get("JourNote", "week_start", fallback="1")
	if "0" <= week_start <= "6":
		week_start = int(week_start)
except configparser.Error as e:
	dialog(msgbox(e), title="Config error")

if args.today:
	edit_file(time.strftime(date_format) + file_ext)
	sys.exit(0)
elif args.notes:
	edit_file(notes_file + file_ext)
	sys.exit(0)

while True:
	code, output = dialog(
		menu("Pick an operation", main_menu),
		title="Main menu", no_tags=None,
		ok_label="Go", cancel_label="Quit")

	if code == DLG_CANCEL:
		code, output = dialog(yesno("Quit JourNote?"),
			yes_label="Quit", no_label="Stay")
		if code == DLG_YES:
			break
	elif output == "today":
		edit_file(time.strftime(date_format) + file_ext)
	elif output == "cal":
		code, output = dialog(calendar("Pick a date"),
			date_format=date_format, week_start=week_start,
			iso_week=None, ok_label="Edit", cancel_label="Back")
		if code == DLG_OK:
			edit_file(output + file_ext)
	elif output == "notes":
		edit_file(notes_file + file_ext)
	elif output == "recent":
		files = glob.glob("*" + file_ext)
		if notes_file in files:
			files.remove(notes_file)
		files.sort(reverse=True)

		entries = file_menu(files[0:file_limit])
		
		if len(entries) > 0:
			handle_recent(entries)
		else:
			dialog(msgbox("No recent entries"))
	elif output == "search":
		if shutil.which("grep") == None:
			print("Please install the grep utility.",
				file=sys.stderr)
			continue

		code, output = dialog(inputbox("Search for"),
			ok_label="Find", cancel_label="Back")
		if code != DLG_OK:
			continue
		
		files = find_in_files(output)
		if len(files) > 0:
			entries = file_menu(files[0:file_limit])
			handle_search(output, entries)
		else:
			dialog(msgbox(output + " not found"))
	elif output == "archive":
		files = glob.glob("*" + file_ext)
		if notes_file in files:
			files.remove(notes_file)
		files.sort()
		
		entries = file_checklist(files[0:file_limit])
		if len(entries) == 0:
			dialog(msgbox("No available entries"))
			continue
		
		code, output = dialog(
			checklist("Entries to archive", entries),
			title="Oldest entries", no_tags=None,
				hline="Showing oldest " + str(file_limit),
				ok_label="Move", cancel_label="Back")
		if code == DLG_OK:
			counter = 0
			try:
				os.makedirs(archive_dir, exist_ok=True)
				for i in output:
					shutil.move(i, archive_dir)
					counter += 1
			except OSError as e:
				dialog(msgbox(e), title="File move failed")
			dialog(msgbox(str(counter) + " entries moved"))
	elif output == "opt":
		handle_options()
	elif output == "version":
		about = about_text.format(version_string=version_string)
		dialog(msgbox(about), title="About JourNote")
